/* Copyright (C) 2016-2018 RealVNC Ltd.  All Rights Reserved.
 */

#include <vnccommon/EventHandleTogglable.h>

#include <fcntl.h>
#include <unistd.h>

using namespace vnccommon;

class EventHandleTogglable::InnerImpl : public EventHandleTogglable::Inner
{
    VNCCOMMON_DISALLOW_COPYING(InnerImpl)

public:
    InnerImpl()
        :   mFlag(false)
    {
        {
            const int result = ::pipe(mFds);

            if(result != 0)
            {
                throw std::logic_error("Failed to allocate pipe");
            }
        }

        const int flags = ::fcntl(mFds[0], F_GETFL);
        const int result = ::fcntl(mFds[0], F_SETFL, flags | O_NONBLOCK);
        if (result != 0)
        {
            closePipe();
            throw std::logic_error("Failed to set O_NONBLOCK on pipe.");
        }
    }

    void set()
    {
        if(mFlag)
        {
            // Already set
            return;
        }

        mFlag = true;

        int byte = 0;
        if(::write(mFds[1], &byte, 1) != 1)
        {
            throw std::logic_error("Failed to write to pipe");
        }
    }

    void unset()
    {
        if(!mFlag)
        {
            // Already unset
            return;
        }

        mFlag = false;

        int byte;
        if(::read(mFds[0], &byte, 1) != 1)
        {
            throw std::logic_error("Failed to read from pipe");
        }
    }

    EventHandle getEventHandle() const
    {
        return mFds[0];
    }

    DirectedEventHandle getDirectedEventHandle() const
    {
        return DirectedEventHandle(getEventHandle(), true, false);
    }

    virtual ~InnerImpl()
    {
        closePipe();
    }

    static InnerImpl& get(Inner& impl)
    {
        return *static_cast<InnerImpl*>(&impl);
    }

private:
    void closePipe()
    {
        ::close(mFds[0]);
        ::close(mFds[1]);
    }

private:
    int mFds[2];
    bool mFlag;
};

EventHandleTogglable::EventHandleTogglable()
    :   mImpl(new InnerImpl())
{
}

void EventHandleTogglable::set()
{
    InnerImpl::get(*mImpl).set();
}

void EventHandleTogglable::unset()
{
    InnerImpl::get(*mImpl).unset();
}

EventHandle EventHandleTogglable::getEventHandle() const
{
    return InnerImpl::get(*mImpl).getEventHandle();
}

DirectedEventHandle EventHandleTogglable::getDirectedEventHandle() const
{
    return InnerImpl::get(*mImpl).getDirectedEventHandle();
}

